function [B] = easi(X, k, res, varargin)
% EASI    Use the EASI algorithm to learn the weights of the
% matrix B for ICA.
%
% B = easi(X, k, options)
%   X is the set of input vectors.  Each column of X is one sample.
%   k is the number of independent components to extract.
%   B is the seperating matrix, Y = BX are the independent signals
%
% options (specified by key/value pairs)
%   'rate' = 0.001      the learning rate (initial)
%   'B' = rand          the initial seperating matrix
%   'niter' = 1         the number of iterations to run
%   'g' = t             the 'g' independence forcing function
%   'h' = tanh(t)       the 'h' independence forcing function

%
% Applied Machine Learning class - Learning Algorithms and Systems
% Laboratory, EPFL
% Demo Code and Images by Basilio Noris, 2010
%
% Original ICA Code by David Gleich, 2003
% http://www.stanford.edu/~dgleich/projects/pca_neural_nets_website/
%

% dimensionality assessment
[m n] = size(X);
if (k > m)
    error('Cannot extra more sources than sensors.');
end;

% subtract the mean
X = X - repmat(mean(X,2), 1, n);

%'B', rand(k,m).*2-1, ...
% options
options = struct(...
    'rate', 0.001, ...
    'B', rand(k,m).*2-1, ...
    'niter', 1, ...
    'g', inline('t'), ...
    'h', inline('tanh(t)'), ...
    'saveSteps', false);
options = getopt(options, varargin);

B = options.B;
u = options.rate;
g = options.g;
h = options.h;
saveSteps = options.saveSteps;

success = 0;
iter = 1;
imgnb = 1;


perm = randperm(n);
for niter=1:options.niter
    Bold = B;
    
    % update B, S
    for ii = 1:n
        index = perm(ii);
        y = B*X(:, index);
        B = B - u*(y*y' - eye(k,k) + g(y)*h(y') - h(y)*g(y'))*B;
      %  B(:,1)=B(:,1)./norm(B(:,1));
      %  B(:,2)=B(:,2)./norm(B(:,2));
        
        if (any(~isfinite(B)))
            warning(sprintf('Lost convergence at iterator %i; lower learning rate?', iter));
            success = 11;
            break;
        end;
        if(mod(ii,1000) == 1)
            Y = B*X;
            e1 = mat2gray(reshape(Y(1,:), res(1), res(2)));
            e2 = mat2gray(reshape(Y(2,:), res(1), res(2)));
            e1 = (e1-min(e1(:)))* 255./(max(e1(:))-min(e1(:)));
            e2 = (e2-min(e2(:)))* 255./(max(e2(:))-min(e2(:)));
            figure(2);clf;
            montage([e1./255 e2./255]);
            drawnow;
            
            if mod(ii,100) == 1
            figure(20); clf; hold on;
            n1=norm(B(:,1));
            n2=norm(B(:,2));
            plot([0 B(1,1)/n1],[0 B(2,1)/n1],'r');
            plot([0 B(1,2)/n2],[0 B(2,2)/n2],'b');
            legend('S1','S2');
            axis([-1 1 -1 1]);
            drawnow
            end;
            
            if(saveSteps)
                filename = sprintf('mix-iter%d.png',imgnb);
                imwrite([e1./255 e2./255], filename, 'png');
                imgnb = imgnb+1;                
            end
        end

        iter = iter + 1; 
        progress = progressbar((niter-1)/options.niter + (1/options.niter)*(ii/n));
        if(progress)
            break;
        end;
        

        
    end;
   
  
 
    
    if (success == 1)
        break;
    end;
end;

function properties = getopt(properties,varargin)
%GETOPT - Process paired optional arguments as 'prop1',val1,'prop2',val2,...
%
%   getopt(properties,varargin) returns a modified properties structure,
%   given an initial properties structure, and a list of paired arguments.
%   Each argumnet pair should be of the form property_name,val where
%   property_name is the name of one of the field in properties, and val is
%   the value to be assigned to that structure field.
%
%   No validation of the values is performed.
%%
% EXAMPLE:
%   properties = struct('zoom',1.0,'aspect',1.0,'gamma',1.0,'file',[],'bg',[]);
%   properties = getopt(properties,'aspect',0.76,'file','mydata.dat')
% would return:
%   properties =
%         zoom: 1
%       aspect: 0.7600
%        gamma: 1
%         file: 'mydata.dat'
%           bg: []
%
% Typical usage in a function:
%   properties = getopt(properties,varargin{:})

% Function from 
% http://mathforum.org/epigone/comp.soft-sys.matlab/sloasmirsmon/bp0ndp$crq5@cui1.lmms.lmco.com

% dgleich 
% 2003-11-19
% Added ability to pass a cell array of properties
if (iscell(varargin{1}))
    varargin = varargin{1};
end;

% Process the properties (optional input arguments)
prop_names = fieldnames(properties);
TargetField = [];
for ii=1:length(varargin)
  arg = varargin{ii};
  if isempty(TargetField)
    if ~ischar(arg)
      error('Propery names must be character strings');
    end
    f = find(strcmp(prop_names, arg), 1);
    if isempty(f)
      error('%s ',['invalid property ''',arg,'''; must be one of:'],prop_names{:});
    end
    TargetField = arg;
  else
    properties.(TargetField) = arg;
    TargetField = '';
  end
end
if ~isempty(TargetField)
  error('Property names and values must be specified in pairs.');
end

